/*
* @author  Umesh 
* @version 1.0
*
* Development Environment         : JDeveloper 2.0
*
* Name of the Application         : BFILESample.java
*
* Creation/ Modification History  :
*
*    Umesh       11-Jan-1999      Creation
*    Sujatha     14-May-2002      Re-certified on Oracle9i JDeveloper
*
*/
package oracle.otnsamples.jdbc.bfile;

// Package containing JDBC classes
import java.sql.Connection;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.PreparedStatement;

//Package for using Streams
import java.io.IOException;
import java.io.InputStream;

//Package for Oracle Extensions to JDBC
import oracle.jdbc.driver.OracleResultSet;
import oracle.jdbc.pool.OracleDataSource;

// Java Utility Classes
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.Enumeration;

import java.awt.Dimension;
import java.awt.Toolkit;

/**
 *  This sample application illustrates accessing of BFILEs using JDBC.
 *
 *  BFILE datatype allows us to store large binary objects (like image files,
 *  executable files) in operating system files outside the database.
 *
 *  This Sample uses JDBC-Thin Driver and JDBC-Streams to access an image file
 *  stored as BFILE column and displays the image in the panel.
 *
 *  The GUI for this sample is handle by the class BFILEFrame. (BFILEFrame.java)
 */
public class BFILESample {

  Connection connection; // Database Connection Object
  BFILEFrame gui;        // GUI handler for this sample

  /**
   *  Constructor. Instantiates GUI.
   */
  public BFILESample() {
    gui = new BFILEFrame(this); // Instantiate GUI
    
    // Diplay the frame in the center of screen
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    Dimension frameSize  = gui.getSize();
    if( frameSize.height > screenSize.height ) {
      frameSize.height = screenSize.height;
    }
    if (frameSize.width > screenSize.width) {
      frameSize.width = screenSize.width;
    }
    gui.setLocation((screenSize.width - frameSize.width) / 2, 
                      (screenSize.height - frameSize.height) / 2);
    gui.setVisible(true);
  }

  /**
   *  Main entry point to the application class. Instantiates the class,
   *  sets up the database connection. Also creation of table BFILE_TABLE and
   *  insertion of an image into the table is done here.
   */
  public static void main(String[] args) {
    BFILESample  rootFrame  = new BFILESample();        // Root Frame
    rootFrame.dbConnection(); // Connects To Database
    if( rootFrame.connection != null ) {
      rootFrame.createTable(); // Creates Table BFILE_TABLE
      rootFrame.insertData();  // Inserts Image Into Table BFILE_TABLE
    }  
  }

  /**
   *  Dispatches the GUI events to the appropriate method, which performs
   *  the required JDBC operations. This method is invoked from the setupListeners
   *  section of BFILEFrame.java
   */
  public void dispatchEvent(String eventName){
    // Dispatch Event to appropriate Methods
    if( eventName.equals("SELECT") )
      selectBFILE();
    else if( eventName.equals("EXIT") )
      exitApplication();
  }

  /**
   * This method reads a properties file which is passed as
   * the parameter to it and load it into a java Properties 
   * object and returns it.
   */
  public static Properties loadParams( String file ) throws IOException {
    // Loads a ResourceBundle and creates Properties from it
    Properties prop = new Properties();
    ResourceBundle bundle = ResourceBundle.getBundle( file );
    Enumeration enum = bundle.getKeys();
    String key = null;
    while( enum.hasMoreElements() ) {
      key = (String)enum.nextElement();
      prop.put( key, bundle.getObject( key ) );
    }
    return prop;
  }

  /**
   * Creates a database connection object using DataSource object. Please
   * substitute the database connection parameters with appropriate values in
   * Connection.properties file
   */
  public void dbConnection() {
    try {
      gui.putStatus("Trying to connect to the Database");

      // Load the properties file to get the connection information
      Properties prop = this.loadParams("Connection");

      // Create a OracleDataSource instance
      OracleDataSource ods = new OracleDataSource();

      // Sets the driver type
      ods.setDriverType("thin");

      // Sets the database server name
      ods.setServerName((String)prop.get("HostName"));

      // Sets the database name
      ods.setDatabaseName((String)prop.get("SID"));

      // Sets the port number
      ods.setPortNumber(new Integer((String)prop.get("Port")).intValue());

      // Sets the user name
      ods.setUser((String)prop.get("UserName"));

      // Sets the password
      ods.setPassword((String)prop.get("Password"));

      // Create a connection  object
      connection = ods.getConnection();

      // Sets the auto-commit property for the connection to be false.
      connection.setAutoCommit(false);

      gui.putStatus(" Connected to " + prop.get("SID") +
                    " Database as " + prop.get("UserName"));


    } catch(SQLException ex) { // Trap SQL errors
        System.out.println(
                     "Error in Connecting to the Database "+'\n'+ex.toString());
    } catch(IOException ex) { // Trap SQL errors
        System.out.println(
                     "Error in reading the properties file "+'\n'+ex.toString());
    }
  }

  /**
   * Method to drop table BFILE_TABLE if it already exists. Then it creates the
   * table BFILE_TABLE (BFILE_COLUMN BFILE). This Table is used to store a row
   * with image file as a BFILE Column in that row.
   */
  public void createTable() {
    try {
      // Create a Statement Object
      Statement stmt = connection.createStatement();

      // Use the Statement Object to execute query to DROP table BFILE_TABLE
      stmt.executeQuery(" DROP TABLE BFILE_TABLE ");

      stmt.close(); // Close the Statement Object

    } catch( SQLException ex ) { //Trap SQL Errors
      gui.putStatus(" Error while Dropping Table " + ex.toString());
    }
    try {
      // Create a Statement Object
      Statement stmt = connection.createStatement();

      // Use the Statement Object to execute query to CREATE table BFILE_TABLE
      stmt.executeQuery(" CREATE TABLE BFILE_TABLE(BFILE_COLUMN BFILE)");

      stmt.close(); // Close the Statement Object

      gui.putStatus("Created table BFILE_TABLE.  Inserting Image Data..");
    } catch (SQLException ex) { //Trap SQL Errors
      gui.putStatus(" Error while Creating Table " + ex.toString());
    }
  }

  /**
   * Method to insert a single record consisting of an image file ('architect.gif')
   * as BFILE column into the table BFILE_TABLE.
   * Please copy the file 'architect.gif' provided along with the sample to the
   * machine where database is running.
   */
  public void insertData() {
    // Prompt the user to enter the directory Path where he/she has copied
    // the image file architect.gif
    String dirPath = gui.getDirectoryPath();
    
    if( !dirPath.equals("Cancel") ) // If user has entered Directory Path
      try {
        // Prepare a SQL Statement to Create Directory
        PreparedStatement pstmt = connection.prepareStatement(
            "CREATE OR REPLACE DIRECTORY BFILEDIR AS '" + dirPath + "' ");
        pstmt.execute(); // Execute the prepared SQL Statement

        // Insert the Image as a BFILE Column
        pstmt.executeQuery(" INSERT INTO BFILE_TABLE VALUES " +
             "( BFILENAME('BFILEDIR', 'architect.gif'))");
             
        pstmt.close(); // Close the prepared Statement
        
        gui.putStatus("Created and populated BFILE_TABLE\n" +
         "Press SELECT to select image" );
        gui.selectButton.setEnabled(true);
      } catch (SQLException ex)  { // Trap SQL Errors
        gui.putStatus(" Error while inserting BFILE Data " + ex.toString());
      }
    else {
      gui.putStatus(" Exiting the application..");
      exitApplication();
    }
  }

  /**
   * Method to Select the BFILE Locator from the BFILE_TABLE and invokes the
   * displayFile() method to display the image. Note that BFILE Columns in
   * Oracle8i store the locator to the external files and not the actual File Data.
   */
  public void selectBFILE(){
    try {
      // Create Statement Object
      Statement stmt = connection.createStatement();

      // Execute the Query and get BFILE Locator as a result set
      ResultSet rset = stmt.executeQuery("SELECT * FROM BFILE_TABLE ");

      // Note : Using Oracle Extension oracle.sql.BFILE to get BFILE Locator
      oracle.sql.BFILE bfile = null;

      // Loop through the Result Set
      while( rset.next() ) {
        bfile = ((OracleResultSet)rset).getBFILE(1); // Get the BFILE Locator
      }
      gui.putStatus("Retrieved BFILE Locator..");

      rset.close(); // Close the ResultSet
      stmt.close(); // Close the Statement

      // Open the file with openFile Method of oracle.sql.BFILE class
      bfile.openFile();

      // Open the Input Binary Stream with  getBinaryStream method of
      // oracle.sql.BFILE class
      InputStream instream = bfile.getBinaryStream();
      gui.appendStatus("Opened Binary Stream ...");
      gui.displayFile(instream); // Invoke displayFile Method to display the image

    } catch( SQLException ex ) { // Trap SQL errors
      gui.putStatus(" Error while selecting BFILE Locator " + ex.toString());
    }
  }

  /**
  * Close connection and exit application
  **/
  public void exitApplication(){
    try {
      if (connection != null)
        connection.close();
    } catch( SQLException ex ){
      gui.putStatus("Error while closing Connection \n" + ex.toString());
    }
    System.exit(0);
  }
}
