/*
 * @author  Reghu
 * @version 1.0
 *
 * Development Environment        :  JDeveloper 2.0
 * Name of the Application        :  ColumnTypeSample.java
 * Creation/Modification History  :
 *
 *    Reghu       14-Jan-1999      Created
 *    Sujatha     14-May-2002      Re-certified on Oracle9i JDeveloper 
 *
 */
package oracle.otnsamples.jdbc.columntype;

// Package for using Streams
import java.io.IOException;

// Package for JDBC Classes
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.Connection;
import java.sql.Statement;
import java.sql.Types;
import java.sql.ResultSet;
import oracle.jdbc.driver.OraclePreparedStatement; 
import oracle.jdbc.pool.OracleDataSource;

// Java Utility Classes
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.Enumeration;

import java.awt.Dimension;
import java.awt.Toolkit;

/**
 *
 * This sample illustrates the Column Type Specification performance extension
 * of Oracle JDBC drivers. Column Type Specification is a feature that improves
 * performance of selecting rows, by specifying the type of the column.
 * By specifying the column type a network roundtrip to obtain the column type
 * is avoided.
 *
 * This sample retrieves 10 rows from a table both by specifying and not
 * specifying the column type. Both types of accesses are timed.
 *
 * The GUI for this sample is handled in ColumnTypeFrame.java
 *
 */
public class ColumnTypeSample {
   // Database Connection Object
   Connection  connection = null;

   // GUI handler for this sample
   ColumnTypeFrame gui;

   int numTotalRows ;   // Total number of rows
   int numFetchedRows ; // Number of rows fetched

  /**
   * Constructor, instantiates GUI.
   */
  public ColumnTypeSample() {
    gui = new ColumnTypeFrame(this); // Instantiate GUI
    
    // Diplay the frame in the center of screen
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    Dimension frameSize  = gui.getSize();
    if( frameSize.height > screenSize.height ) {
      frameSize.height = screenSize.height;
    }
    if (frameSize.width > screenSize.width) {
      frameSize.width = screenSize.width;
    }
    gui.setLocation((screenSize.width - frameSize.width) / 2, 
                      (screenSize.height - frameSize.height) / 2);
    
    gui.setVisible(true);
  }

  /**
   * Main entry point of the Class. Instantiates the root frame, then
   * establishes a Database connection and sets numTotalRows to the
   * total number of rows in AVAILABLE_ROOM_TYPES
   */
  public static void main( String args[] ) {
    ColumnTypeSample ColType = new ColumnTypeSample();// Instantiate root frame
    ColType.dbConnection(); // Establishes the DB connection
    if( ColType.connection != null ) {
      ColType.countRows();  // Get the total number of rows in the table
      ColType.gui.buttonStateChange(true);
    }   
  }

  /**
   * Dispatches the GUI events to the appropriate method, which performs
   * the required JDBC operations. This method is invoked when event occurs
   * in the GUI (like Button clicks etc.). This method is invoked from
   * the setupListeners method and run() method of ColTypeFrame.java
   */
  public void dispatchEvent(String eventName) {
    // Dispatch Event
    if( eventName.equals("START SELECT") )
        selectRecords(gui.flag);
    if( eventName.equals("EXIT") )
        exitApplication();
  }

  /**
   * This method reads a properties file which is passed as
   * the parameter to it and load it into a java Properties 
   * object and returns it.
   */
  public static Properties loadParams( String file ) throws IOException {
    // Loads a ResourceBundle and creates Properties from it
    Properties prop = new Properties();    
    ResourceBundle bundle = ResourceBundle.getBundle( file );
    Enumeration enum = bundle.getKeys();
    String key = null;
    while( enum.hasMoreElements() ) {
      key = (String)enum.nextElement();
      prop.put( key, bundle.getObject( key ) );
    }
    return prop;
  }

  /**
   * Creates a database connection object using DataSource object. Please 
   * substitute the database connection parameters with appropriate values in
   * Connection.properties file
   */
  public void dbConnection() {
    try {
      gui.putStatus("Trying to connect to the Database");
 
      // Load the properties file to get the connection information
      Properties prop = this.loadParams("Connection");
 
      // Create a OracleDataSource instance
      OracleDataSource ods = new OracleDataSource();
 
      // Sets the driver type
      ods.setDriverType("thin");
 
      // Sets the database server name
      ods.setServerName((String)prop.get("HostName"));
 
      // Sets the database name
      ods.setDatabaseName((String)prop.get("SID"));

      // Sets the port number
      ods.setPortNumber(new Integer((String)prop.get("Port")).intValue());
 
      // Sets the user name
      ods.setUser((String)prop.get("UserName"));
 
      // Sets the password
      ods.setPassword((String)prop.get("Password"));
 
      // Create a connection  object
      connection = ods.getConnection();
 
      // Sets the auto-commit property for the connection to be false.
      connection.setAutoCommit(false);
 
      gui.putStatus(" Connected to " + prop.get("SID") +
                    " Database as " + prop.get("UserName"));

 
    } catch(SQLException ex) { // Trap SQL errors
        System.out.println(
                     "Error in Connecting to the Database "+'\n'+ex.toString());
    } catch(IOException ex) { // Trap SQL errors
        System.out.println(
                     "Error in reading the properties file "+'\n'+ex.toString());
    }
  }

  /**
   * This method selects records from the database using column type specification
   * If specifyColumnType is true and without specifying column types if
   * specifyColumnType is false.
   */
  public void selectRecords( boolean specifyColumnType ) {
    Integer  timetaken = null; // Holds the execution time
    numFetchedRows = 0;

    // The SQL query for querying 10 rows from the table
    String query = "SELECT standard_rate FROM available_room_types "
                                                       + "WHERE ROWNUM < 11";
    try {
      gui.putStatus("Started selecting rows, Please wait..");

      // Create a PreparedStatement based on the query in query
      PreparedStatement prepare =  connection.prepareStatement(query);

      // Casting the PreparedStatement to OraclePreparedStatement and
      // Specifying the Standard_rate Column type as FLOAT using
      // defineColumnType() method.
      if( specifyColumnType )
        ((OraclePreparedStatement)prepare).defineColumnType(1,Types.VARCHAR);

        // Get time at start of query
        java.util.Date bfdte = new java.util.Date();

        // Execute the Query in query
        ResultSet rst = prepare.executeQuery();

        String temp = null;

        // Populate the ResultSet object
        while( rst.next() ) {
          numFetchedRows++;

          if( specifyColumnType )
            // Column Type specified
            temp = rst.getString(1);
          else
            // Column type not specified
            temp = new Float( rst.getFloat(1) ).toString();
         }

        // Get time at end of query
        java.util.Date afdte = new java.util.Date();

        // Compute time taken
        timetaken = new Integer((int)( afdte.getTime()- bfdte.getTime()));

        gui.displayResult( specifyColumnType, timetaken); //Display the results
        gui.putStatus( numFetchedRows + " Rows Selected" );
        prepare.close(); // Close the PreparedStatement object

    } catch( SQLException ex ) { // Trap SQL Errors
      gui.timer.stop();
      gui.putStatus(" Error in Selecting rows "+'\n'+ ex.toString());
    }
  }

  /**
   * Count the total number of rows in the AVAILABLE_ROOM_TYPES table.
   * Required to maintain the progress bar
   */
  public void countRows() {
    // The SQL query for counting the total number of rows
    // from AVAILABLE_ROOM_TYPES table
    String query = "SELECT COUNT(*) FROM available_room_types "
                                                      + " WHERE ROWNUM < 3000";
    try {
      // Create a SQL statement
      Statement stmt = connection.createStatement();

      // Execute the Query in query
      ResultSet result = stmt.executeQuery( query );

      // Retrieve the results from the resultset
      if(result.next() )
        numTotalRows = result.getInt(1);// Gets the total number of rows

        // Close the statement object, which also closes the resultset
        stmt.close();

    } catch( SQLException ex ) { // Trap SQL Errors
      gui.putStatus(" Error in selecting the rows "+'\n'+ ex.toString());
    }
  }

  /**
   * Closing the Database Connection Object and exit the application
   */
  public void exitApplication() {
    try {
      gui.putStatus("Closing the connection....please wait.....");
      if( connection != null )
        connection.close(); // Closing the connection object.
    } catch( SQLException ex ) {  // Trap SQL Errors
      gui.putStatus(ex.toString());
    }
    System.exit(0); // Exit the aplication
  }
}

