/*
 * @author       Arun
 * @version 1.0
 *
 * Development Environment        :  JDeveloper 2.0
 * Name of the Application        :  DateSample.java
 * Creation/Modification History  :
 *
 *    Arun        30-Dec-1998      Created
 *    Sujatha     14-May-2002      Re-certified on Oracle9i JDeveloper  
 *
 */
package oracle.otnsamples.jdbc.datetype;

// Package containing JDBC classes
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import oracle.jdbc.pool.OracleDataSource;

// Java Utility Classes
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.Enumeration;

import java.awt.Dimension;
import java.awt.Toolkit;

// Package for using Streams
import java.io.IOException;

/**
 * This sample demonstrates the use of JDBC calls to access DATE fields.
 * The user gives a specific departure Date and on pressing SELECT, the application
 * retrieves all flights on that departure date.
 *
 * The GUI part of this sample is handled seperately in DateFrame.java
 */
public class DateSample  {

  Connection connection; // database connection object
  DateFrame gui;         // For Handling the GUI related operations

  /**
  * Constructor which in turn calls the constructor of GUI class
  **/
  public DateSample() {
    gui =new DateFrame(this);
      
    // Diplay the frame in the center of screen
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    Dimension frameSize  = gui.getSize();
    if( frameSize.height > screenSize.height ) {
      frameSize.height = screenSize.height;
    }
    if (frameSize.width > screenSize.width) {
      frameSize.width = screenSize.width;
    }
    gui.setLocation((screenSize.width - frameSize.width) / 2, 
                      (screenSize.height - frameSize.height) / 2);
    gui.setVisible(true);                      
  }

  /**
   *  Main entry point for the class. Instantiates the root frame and
   *  sets up the database connection. Also calls displayDateLimits()
   *  to display the range of dates between which departures are available
   */
  public static void main( String[] args ) {
    DateSample DS = new DateSample();
    DS.dbConnection(); // Establishes the DB connection
    if( DS.connection != null ) {
      DS.displayDateLimits(); // Display the available dates
    }
  }

  /**
   *  Dispatches the GUI events to the appropriate method, which performs
   * the required JDBC operations. This method is invoked when event occurs
   * in the GUI (like table Selection, Button clicks etc.). This method
   * is invoked from the setupListeners section of DateFrame.java
   */
  public void dispatchEvent (String eventName) {

    //  For dispatching the select event
    if( eventName.equals("SELECT") )
      selectRecords();

    // For dispatching the exit event
    if( eventName.equals("EXIT") )
      exitApplication();
  }

  /**
   * This method reads a properties file which is passed as
   * the parameter to it and load it into a java Properties 
   * object and returns it.
   */
  public static Properties loadParams( String file ) throws IOException {
    // Loads a ResourceBundle and creates Properties from it
    Properties prop = new Properties();
    ResourceBundle bundle = ResourceBundle.getBundle( file );
    Enumeration enum = bundle.getKeys();
    String key = null;
    while( enum.hasMoreElements() ) {
      key = (String)enum.nextElement();
      prop.put( key, bundle.getObject( key ) );
    }
    return prop;
  }

  /**
   * Creates a database connection object using DataSource object. Please
   * substitute the database connection parameters with appropriate values in
   * Connection.properties file
   */
  public void dbConnection() {
    try {
      gui.putStatus("Trying to connect to the Database");

      // Load the properties file to get the connection information
      Properties prop = this.loadParams("Connection");

      // Create a OracleDataSource instance
      OracleDataSource ods = new OracleDataSource();

      // Sets the driver type
      ods.setDriverType("thin");

      // Sets the database server name
      ods.setServerName((String)prop.get("HostName"));

      // Sets the database name
      ods.setDatabaseName((String)prop.get("SID"));

      // Sets the port number
      ods.setPortNumber(new Integer((String)prop.get("Port")).intValue());

      // Sets the user name
      ods.setUser((String)prop.get("UserName"));

      // Sets the password
      ods.setPassword((String)prop.get("Password"));

      // Create a connection  object
      connection = ods.getConnection();

      // Sets the auto-commit property for the connection to be false.
      connection.setAutoCommit(false);

      gui.putStatus(" Connected to " + prop.get("SID") +
                    " Database as " + prop.get("UserName"));


    } catch(SQLException ex) { // Trap SQL errors
        System.out.println(
                     "Error in Connecting to the Database "+'\n'+ex.toString());
    } catch(IOException ex) { // Trap SQL errors
        System.out.println(
                     "Error in reading the properties file "+'\n'+ex.toString());
    }
  }

  /**
   * Queries the database for the minimum departure date and maximum
   * departure date.
   */
  public void displayDateLimits() {
    ResultSet result;
    String message = "";
    try {
      // PreparedStatement is used to form queries during runtime
      PreparedStatement ps = connection.prepareStatement(
         "SELECT MIN(departure_date), MAX(departure_date) FROM flight_departures");

      result = ps.executeQuery(); // Execute the query

      result.next(); // Goto the first record in the result-set

      // Retrieve the dates from the resultset record
      java.util.Date minDate, maxDate;
      minDate = result.getDate(1);
      maxDate = result.getDate(2);

      gui.displayDateLimits( minDate, maxDate);
      result.close(); // Close the result set
      ps.close();     // Close the statement object

    } catch( SQLException ex ) { // Trap SQL errors
      gui.putStatus(" Error in Querying: ");
      gui.appendStatus(" " + ex.toString());
    }
  }

  /**
   * Queries the database for the departure date entered and displays the
   * result in the text fields.
   */
  void selectRecords() {
    gui.putStatus("Accessing database. Please Wait ....");
    gui.statusField.validate();
    ResultSet result;
    try {
      // PreparedStatement is used to form queries during runtime
      // This query obtains the departure time in a "hh:mm:ss" format
      // from the departure_time field, which is a number
      PreparedStatement ps = connection.prepareStatement(
                " SELECT id ,"+
                " SUBSTR( LPAD(departure_time,6,0),1,2) || ':' ||" +
                " SUBSTR( LPAD(departure_time,6,0),3,2) time, " +
                " flight_status FROM flight_departures" +
                " WHERE TO_CHAR( departure_date,'DD-Mon-YY') = ?" );

      // Bind the statement parameters
      ps.setString(1, gui.depDate.getText());  // Bind departure date

      result = ps.executeQuery();  // Execute the  query

      gui.tableModel.clearTable(); // Clear JTable in GUI

      while( result.next() ) { // Point result set to next row
         // Retrieve column values for this row
        String code    = result.getString(1);
        String name    = result.getString(2);
        String partner = result.getString(3);

        gui.addToJTable( code, name, partner); // Update GUI
      }

      result.close();               // close the result set
      ps.close();                   // close the statement after query
      gui.flightPane.repaint();
      displayDateLimits();
    } catch( SQLException ex ) {
      gui.putStatus(" Error in Querying: ");
      gui.appendStatus(" " + ex.toString() );
    }
  }

  /**
   *  Closes the connection and exits from the program when exit button is
   *  pressed
   */
  public void exitApplication(){
    if (connection != null) {
      try {
        connection.close();  // Close the connection
      } catch( Exception ex ) {
        gui.putStatus(" Error in Closing the connection: ");
        gui.appendStatus(" "+ex.toString());
      }
    }
    System.exit(0);
  }
}




