/*
 * @author  rsundara.in
 * @version 1.0
 *
 * Development Environment        :  Oracle9i JDeveloper
 * Name of the Application        :  LongRawSample.java
 * Creation/Modification History  :
 *
 * rsundara.in       28-Dec-1998      Created
 * neshastr          22-May-2002      Re-certified on Oracle9i JDeveloper
 */
package oracle.otnsamples.jdbc.longraw;

import java.util.Enumeration;
import java.util.ResourceBundle;
import java.util.Properties;
import java.io.IOException;
import java.io.InputStream;
import java.io.FileOutputStream;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.sql.SQLException;

import oracle.jdbc.pool.OracleDataSource;

/**
  * This sample illustrates accessing  Oracle LONGRAW datatype using JDBC.
  * Long raw column is normally used for storing large binary data like image,
  * sound files upto a maximum of 2GB.
  *
  * In this sample, image files stored in a LONGRAW column are retrieved
  * and displayed on the front end interface. The  table AIRLINE_LONG_RAW_TABLE
  * is used to store Long Raw Data
  *
  * AIRLINE_LONG_RAW_TABLE is created and populated with sample image files,
  * by running the populateTable.java which is present in the install directory.
  *
  * The gui for this sample is handled in LongRawFrame.java
  *
  * The sample uses the JDBC-Thin driver. JDBC-OCI driver can be used after
  * modifying the database URL in the dbConnection method.
  */
public class LongRawSample {

  Connection  connection  = null;     // Database Connection Object
  LongRawFrame gui;                   // gui handler for this sampl

  /**
   *  Constructor. Instantiates gui.
   */

  public LongRawSample() {
    try {
      gui = new LongRawFrame(this);  // Instantiate gui
      gui.setVisible(true);
    } catch (Exception e) {  //Trap general Errors
      e.printStackTrace();
    }
  }

  /**
   *  Main entry point for the class. Instantiates class and  sets up the
   *  database connection.The selection of airline names are done here.
   */

  public static void main(String[] args)throws SQLException{
    LongRawSample LRS = new LongRawSample();// Instantiate application class
    LRS.dbConnection();                     // Setup the db connection
    LRS.selectAirlines();                   // Fills the JTable with records
  }

  /**
   * Dispatches the gui events to the appropriate method, which performs
   * the required JDBC operations. This method is invoked when event occurs
   * in the gui (like table Selection, Button clicks etc.). This method
   * is invoked from the setupListeners section of LongRawFrame.java
   */
  public void dispatchEvent(String eventName) {

    // Dispatch Event
    if( eventName.equals("SELECT") ) {
      // Get the user selected airline code
      int selectedRow=gui.table.getSelectedRow();
      String code=(String)gui.tablemodel.getValueAt(selectedRow,0);

      // Retrieve the gif file from the  LONGRAW column in the database
      String nameOfGif=getLogo(code);

      gui.drawLogo(nameOfGif); // Draw the image
    }
    if( eventName.equals("EXIT") )
      exitApplication();
  }

  /**
   * This method reads a properties file which is passed as
   * the parameter to it and load it into a java Properties
   * object and returns it.
   */
  
  public static Properties loadParams(String file) throws IOException {
    
    // Loads a ResourceBundle and creates Properties from it
    Properties prop = new Properties();
    ResourceBundle bundle = ResourceBundle.getBundle(file);
    Enumeration enum = bundle.getKeys();
    String key = null;
    while(enum.hasMoreElements()) {
      key = (String)enum.nextElement();
      prop.put(key, bundle.getObject(key));
    }
    return prop;
  }
  
  /**
   * Creates a database connection object using DataSource object. Please 
   * substitute the database connection parameters with appropriate values in
   * Connection.properties file
   */
  public void dbConnection() {
    try {
      gui.putStatus("Trying to connect to the Database");
 
      // Load the properties file to get the connection information
      Properties prop = this.loadParams("Connection");
 
      // Create a OracleDataSource instance
      OracleDataSource ods = new OracleDataSource();
 
      // Sets the driver type
      ods.setDriverType("thin");
 
      // Sets the database server name
      ods.setServerName((String)prop.get("HostName"));
 
      // Sets the database name
      ods.setDatabaseName((String)prop.get("SID"));
 
      // Sets the port number
      ods.setPortNumber(new Integer((String)prop.get("Port")).intValue());
 
      // Sets the user name
      ods.setUser((String)prop.get("UserName"));
 
      // Sets the password
      ods.setPassword((String)prop.get("Password"));
      connection=ods.getConnection();
      gui.putStatus(" Connected to " + prop.get("SID") +
                    " Database as " + prop.get("UserName"));
 
    } catch(SQLException ex) { // Trap SQL errors
        System.out.println(
                     "Error in Connecting to the Database "+'\n'+ex.toString());
    } catch(IOException ex) { // Trap SQL errors
        System.out.println(
                     "Error in reading the properties file "+'\n'+ex.toString());
    } 
  }
  
  /**
   * This method retrieves the logo of the selected airline from the long raw
   * column in the table.
   */
  public String getLogo(String lcode){
    String name         = null;
    InputStream gifdata = null;
    try{
      // Query the logo of the selected airline
      String query = "SELECT name,airline_logo FROM airline_long_raw_table WHERE code="
                      + "'" +lcode+ "'";
      // Create a statement object for executing the query
      Statement stmt = connection.createStatement();

      // Obtain the result-set for the selected airline record
      ResultSet result = stmt.executeQuery(query);

      if( result.next() ) {

        // Fetch column values
        name = result.getString(1); // Obtain the airline code

        // LONGRAW data can be accessed in two ways:
        // 1) By retrieving all the data at once (using getBytes method)
        // 2) By using streams. The LONGRAW data is made available to the program
        // as a stream, and the data can be retrieved chunk by chunk, which is
        // more eficient in terms of memory usage
        // In this sample we illustrate retrieval using streams method.
        gifdata = result.getBinaryStream(2);

        // Write the byte array into a local file
        FileOutputStream file=new FileOutputStream(name+".gif");

        int chunk=0;
        while( (chunk = gifdata.read()) != -1)
          file.write(chunk);
        file.flush();
        file.close();
        stmt.close();  //closing the statement object
        gui.putStatus("Image obtained from database");
      }
    } catch( Exception ex ) { //Trap IO Errors
     gui.putStatus(ex.toString());
   }
    return name;
  }

  /**
   * This method  selects the airline records from the airline_long_raw_table
   * and fills the front end JTable with the records.
   */
  public void selectAirlines() {
    ResultSet result = null;
    try {
      // Create a statement for executing the query
      Statement statement = connection.createStatement();

      // The query string
      String qry = "SELECT code,name,partner FROM airline_long_raw_table";

      // Execute the query and obtain the result set.
      result = statement.executeQuery(qry);
      gui.tablemodel.clearTable(); // Clear JTable in gui
      while (result.next()) { // Point result set to next row
        // Retrieve column values for this row
        String code = result.getString(1);
        String name = result.getString(2);
        String partner = result.getString(3);
        gui.addToJTable(code, name, partner); // Update gui
      }
      statement.close();   // Close the statement object
    } catch(Exception ex){ // Trap SQL errors
      gui.putStatus("Error in querying from Database "+'\n'+ ex.toString());
    }
  }

  /**
   *  This method closes the connection object before exiting the application &
   *  does a descent exit
   */
  public void exitApplication() {
    if( connection != null ) {
      try {
        connection.close();             
      } catch (SQLException ex) {
        gui.putStatus(" Error in closing the connection: ");
        gui.appendStatus(" "+ ex.toString());
      }
    }
    System.exit(0);
  }
}