
/*
 * @author  Reghu
 * @version 1.0
 *
 * Development Environment        :  Oracle 9i JDeveloper
 * Name of the Application        :  RowPrefetchSample.frame
 * Creation/Modification History  :
 *
 *    rkrishna.in       11-Jan-1999      Created
 *    neshastr          22-May-2002      Modified       Added code to center align the UI frame, updated to reflect new coding standards
 *													        changed package structure
 *
 *
 */
package oracle.otnsamples.jdbc.prefetch;
import java.awt.Color;
import java.awt.GridBagLayout;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.WindowEvent;
import javax.swing.Timer;
import javax.swing.JLabel;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JProgressBar;
import javax.swing.JTextField;

/**
 * This class handles the gui for the RowPrefetchSample application.
 */
public class RowPrefetchFrame extends JFrame implements Runnable {
  
  
  // The main application class which uses this frame
  RowPrefetchSample parent;
  
  Thread fetchThread = null; //Thread Object
  
  
  //Buttons for specifying the operations to be performed
  JButton prefetch200, prefetch100, exit, normal;
  
  
  //Textfields labels for displaying the output and status
  JTextField normaltime, fetchtime100, fetchtime200, statusbar;
  
  JLabel timeLabel;
  
  
  //To illustrates the progress of current fetch
  JProgressBar progressbar;
  
  Timer timer;
  
  int prefetchValue = 10; // Holds the row prefetch value. By default it is 10
  
  
  //Layout manager object
  GridBagLayout gridBagLayout1 = new GridBagLayout();
  
  /**
   * Constructor, initializes and sets up listeners for the gui components of
   * the application
   **/
  
  public RowPrefetchFrame(RowPrefetchSample parent) {
    try {
      this.parent = parent;
      jbInit(); // Other gui components initializations
      setupListeners(); // Sets up the event listeners and actions
      positionScreen();
    }
    catch(Exception ex) {
      putStatus("Error..!!" + '\n' + ex.toString());
    }
  }
  
  /**
   * Start the fetchThread thread to select records from the database
   **/
  
  public void fetchThreadStart(int value) {
    prefetchValue = value;
    if(fetchThread == null) {
      fetchThread = new Thread(this, "RowPrefetch");
      fetchThread.start();
    }
  }
  
  /**
   *  Method to display the results in the text field based on the Batch size
   **/
  
  public void displayResult(int prfetchValue, Integer timetaken) {
    switch(prfetchValue) {
      case 10:
        normaltime.setText(timetaken.toString());
        break;
      
      case 100:
        fetchtime100.setText(timetaken.toString());
        break;
      
      case 200:
        fetchtime200.setText(timetaken.toString());
        break;
    }
  }
  
  /**
   * This methods is called by updateThread Thread class. @see java.lang.Thread
   * Here the run() method is implemented to call dispatchEvent() method of
   * RowPrefetchSample class
   **/
  
  public void run() {
    buttonStateChange(false);
    
    // Dispatch event
    parent.dispatchEvent("START FETCH");
    buttonStateChange(true);
    fetchThread = null;
  }
  
  /**
   * TimeListener class is implemented as an inner class. The actionPerformed()
   * method in this TimerListener class is called each time the Timer "goes off".
   **/
  
  class TimerListener implements ActionListener {
    
    public void actionPerformed(ActionEvent evt) {
      progressbar.setMaximum(parent.numTotalRows);
      
      //Sets the current value of fetchedRows to the Progressbar
      progressbar.setValue(parent.numFetchedRows);
      if(parent.numFetchedRows == parent.numTotalRows) {
        timer.stop();
        progressbar.setValue(progressbar.getMinimum());
      }
    }
  }
  
  /**
   * This method changes the Button state based on the value of state
   **/
  
  public void buttonStateChange(boolean state) {
    normal.setEnabled(state);
    prefetch100.setEnabled(state);
    prefetch200.setEnabled(state);
  }
  
  /**
   * Sets up the listeners that respond to user actions like button press and
   * performing corresponding actions
   **/
  
  public void setupListeners() {
    
    //Setup the "WITHOUT PRE-FETCH" button Listener
    normal.addActionListener(new java.awt.event.ActionListener()  {
      
      public void actionPerformed(ActionEvent evt) {
        if(evt.getActionCommand() == "NORMAL") {
          normaltime.setText("");
          
          // Start a new thread with Row prefetch size = 10
          fetchThreadStart(10);
          
          // Start the Timer to initialize the Progressbar
          timer.start();
        }
      }
    });
    
    //Setup the "WITH ROW-PREFETCH=100" button Listener
    prefetch100.addActionListener(new java.awt.event.ActionListener()  {
      
      public void actionPerformed(ActionEvent evt) {
        if(evt.getActionCommand() == "Prefetch100") {
          fetchtime100.setText("");
          
          // Start a new thread with Row prefetch size = 100
          fetchThreadStart(100);
          
          //Start the Timer to initialize the Progressbar
          timer.start();
        }
      }
    });
    
    //Setup the "WITH ROW-PREFETCH=200" button Listener
    prefetch200.addActionListener(new java.awt.event.ActionListener()  {
      
      public void actionPerformed(ActionEvent evt) {
        if(evt.getActionCommand() == "Prefetch200") {
          fetchtime200.setText("");
          
          // Start a new thread with Row prefetch size = 200
          fetchThreadStart(200);
          
          //Start the Timer to initialize the Progressbar
          timer.start();
        }
      }
    });
    
    //Setup the "EXIT" button Listener
    exit.addActionListener(new java.awt.event.ActionListener()  {
      
      public void actionPerformed(ActionEvent evt) {
        if(evt.getActionCommand() == "EXIT") {
          parent.dispatchEvent("EXIT");
        }
      }
    });
  }
  
  /**
   * Status field message display handlers
   **/
  
  public void clearStatus(String status) { //Clear the statusbar
    statusbar.setText("");
  }
  
  public void putStatus(String status) { //Puts status on statusbar
    statusbar.setText(status);
    statusbar.setScrollOffset(0);
  }
  
  public void appendStatus(String status) { //Appends status to statusbar
    statusbar.setText(statusbar.getText() + "\n" + status);
    statusbar.setScrollOffset(0);
  }
  
  /**
   * Initialise and setup the gui components and Layout of the Application.
   **/
  
  public void jbInit() throws Exception {
    prefetch200 = new JButton();
    prefetch100 = new JButton();
    exit = new JButton();
    normal = new JButton();
    statusbar = new JTextField();
    normaltime = new JTextField();
    fetchtime100 = new JTextField();
    fetchtime200 = new JTextField();
    timeLabel = new JLabel();
    progressbar = new JProgressBar();
    timer = new Timer(200, new TimerListener());
    this.setTitle("Illustrating ROW PRE-FETCH Using JDBC");
    this.getContentPane().setLayout(gridBagLayout1);
    this.setSize(new Dimension(575, 360));
    this.addWindowListener(new java.awt.event.WindowAdapter()  {
      
      public void windowClosing(WindowEvent e) {
        parent.dispatchEvent("EXIT");
      }
    });
    progressbar.setToolTipText("Progress Bar");
    progressbar.setMinimum(0);
    progressbar.setForeground(new Color(0, 0, 244));
    progressbar.setValue(0);
    progressbar.setBorderPainted(true);
    exit.setText("EXIT");
    exit.setRolloverEnabled(true);
    exit.setFont(new Font("Dialog", 1, 16));
    exit.setToolTipText("Press to Exit the Application");
    statusbar.setBackground(Color.lightGray);
    statusbar.setFont(new Font("Monospaced", 0, 16));
    statusbar.setToolTipText("Status Bar");
    statusbar.setEditable(false);
    normaltime.setHorizontalAlignment(JTextField.CENTER);
    normaltime.setBackground(Color.gray);
    normaltime.setFont(new Font("Dialog", 1, 16));
    normaltime.setBorder(null);
    normaltime.setEditable(false);
    fetchtime200.setHorizontalAlignment(JTextField.CENTER);
    fetchtime200.setBackground(Color.gray);
    fetchtime200.setFont(new Font("Dialog", 1, 16));
    fetchtime200.setBorder(null);
    fetchtime200.setEditable(false);
    normal.setText("DEFAULT ROW PRE-FETCH");
    normal.setActionCommand("NORMAL");
    normal.setRolloverEnabled(true);
    normal.setFont(new Font("Dialog", 1, 16));
    normal.setToolTipText("Press to Select Rows without Row Pre-fetch");
    normal.setEnabled(false);
    prefetch200.setEnabled(false);
    timeLabel.setText("Time Taken (ms)");
    timeLabel.setFont(new Font("Dialog", 1, 16));
    fetchtime100.setHorizontalAlignment(JTextField.CENTER);
    fetchtime100.setBackground(Color.gray);
    fetchtime100.setFont(new Font("Dialog", 1, 16));
    fetchtime100.setBorder(null);
    fetchtime100.setEditable(false);
    prefetch100.setText("WITH ROW  PRE-FETCH = 100");
    prefetch100.setActionCommand("Prefetch100");
    prefetch100.setRolloverEnabled(true);
    prefetch100.setFont(new Font("Dialog", 1, 16));
    prefetch100.setEnabled(false);
    prefetch100.setToolTipText("Press to Select Rows with Row-prefetch 100");
    prefetch200.setText("WITH ROW  PRE-FETCH = 200");
    prefetch200.setActionCommand("Prefetch200");
    prefetch200.setRolloverEnabled(true);
    prefetch200.setFont(new Font("Dialog", 1, 16));
    prefetch200.setToolTipText("Press to Select Rows with Row Pre-fetch 200");
    this.getContentPane().add(prefetch200, new GridBagConstraints(0, 5, 1, 2, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(18, 9, 0, 0), 101, 20));
    this.getContentPane().add(prefetch100, new GridBagConstraints(0, 3, 1, 2, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(17, 12, 0, 0), 102, 19));
    this.getContentPane().add(progressbar, new GridBagConstraints(0, 7, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(13, 10, 0, 0), 205, 14));
    this.getContentPane().add(exit, new GridBagConstraints(1, 7, 2, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(0, 18, 0, 3), 77, 13));
    this.getContentPane().add(statusbar, new GridBagConstraints(0, 8, 3, 1, 1.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(10, 11, 1, 1), 528, 48));
    this.getContentPane().add(normaltime, new GridBagConstraints(1, 1, 2, 1, 1.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 16, 0, 0), 147, 8));
    this.getContentPane().add(fetchtime100, new GridBagConstraints(1, 3, 2, 1, 1.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(24, 16, 0, 0), 147, 8));
    this.getContentPane().add(fetchtime200, new GridBagConstraints(1, 5, 2, 1, 1.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(24, 16, 0, 2), 145, 9));
    this.getContentPane().add(timeLabel, new GridBagConstraints(1, 0, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(4, 41, 0, 0), 6, 12));
    this.getContentPane().add(normal, new GridBagConstraints(0, 0, 1, 3, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(28, 13, 0, 0), 102, 20));
  }
  
  /**
   * Repositions the frame to the center of the window
   */
  
  private void positionScreen() {
    
    // Get the screen size
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    
    // Get the frame size
    Dimension frameSize = getSize();
    if(frameSize.height > screenSize.height) {
      frameSize.height = screenSize.height;
    }
    if(frameSize.width > screenSize.width) {
      frameSize.width = screenSize.width;
    }
    
    // Find the center of the screen and reposition the frame
    setLocation((screenSize.width - frameSize.width) / 2, (screenSize.height - frameSize.height) / 2);
  }
}
