
/*
* @author  Reghu
* @version 1.0
*
* Development Environment        :  Oracle 9i JDeveloper
* Name of the Application        :  RowPrefetchSample.java
* Creation/Modification History  :
*
*    rkrishna.in       11-Jan-1999      Created
*    neshastr          22-May-2002      Modified       Changed database connection loolup code,updated to reflect new coding standards
*													   changed package structure
*/
package oracle.otnsamples.jdbc.prefetch;
import java.util.Enumeration;
import java.util.ResourceBundle;
import java.util.Properties;
import java.io.IOException;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;

import oracle.jdbc.driver.OraclePreparedStatement;
import oracle.jdbc.pool.OracleDataSource;

/**
 *
 * This sample illustrates the row-prefetch performance extension of Oracle JDBC
 * drivers. Row-Prefetch allows the user to specify the number of rows to fetch
 * from the result-set in each round-trip to the database.
 *
 * This sample retrieves the rows from the AVAILABLE_ROOM_TYPES table using
 * three different values of row-prefetch. Each retrieval is timed.
 *
 * The gui for this sample is handled in RowPrefetchFrame.java
 *
 */

public class RowPrefetchSample {
  
  
  // Database Connection Object
  Connection connection = null;
  
  
  // gui handler for this sample
  RowPrefetchFrame gui;
  
  int numTotalRows; // Holds the total number of rows in the table
  
  int numFetchedRows; // Holds the number of rows fetched from the table
  
  /**
   * Constructor, instantiates gui.
   **/
  
  public RowPrefetchSample() {
    gui = new RowPrefetchFrame(this); // Instantiate gui
    gui.setVisible(true);
  }
  
  /**
   * Main entry point of the Class. Instantiates the root frame, then
   * establishes a Database connection and sets numTotalRows to the
   * total number of rows in AVAILABLE_ROOM_TYPES
   **/
  
  public static void main(String args[]) {
    RowPrefetchSample Rprefetch = new RowPrefetchSample(); //Instantiate root frame
    Rprefetch.dbConnection(); // Establishes the DB connection
    Rprefetch.countRows(); // Get the total number of rows in the table
    Rprefetch.gui.buttonStateChange(true);
  }
  
  /**
   * Dispatches the gui events to the appropriate method, which performs
   * the required JDBC operations. This method is invoked when event occurs
   * in the gui (like Button clicks etc.). This method is invoked from the
   * setupListeners method and run() method of of RowPrefetchFrame.java
   **/
  
  public void dispatchEvent(String eventName) {
    
    // Dispatch Event
    if(eventName.equals("START FETCH")) {
      prefetchRows(gui.prefetchValue);
    }
    if(eventName.equals("EXIT")) {
      exitApplication();
    }
  }
  
  /**
   * This method reads a properties file which is passed as
   * the parameter to it and load it into a java Properties
   * object and returns it.
   */
  
  public static Properties loadParams(String file) throws IOException {
    
    // Loads a ResourceBundle and creates Properties from it
    Properties prop = new Properties();
    ResourceBundle bundle = ResourceBundle.getBundle(file);
    Enumeration enum = bundle.getKeys();
    String key = null;
    while(enum.hasMoreElements()) {
      key = (String)enum.nextElement();
      prop.put(key, bundle.getObject(key));
    }
    return prop;
  }
  
 /**
   * Creates a database connection object using DataSource object. Please 
   * substitute the database connection parameters with appropriate values in
   * Connection.properties file
   */
  public boolean dbConnection() {
    try {
      gui.putStatus("Trying to connect to the Database");
 
      // Load the properties file to get the connection information
      Properties prop = this.loadParams("Connection");
 
      // Create a OracleDataSource instance
      OracleDataSource ods = new OracleDataSource();
 
      // Sets the driver type
      ods.setDriverType("thin");
 
      // Sets the database server name
      ods.setServerName((String)prop.get("HostName"));
 
      // Sets the database name
      ods.setDatabaseName((String)prop.get("SID"));
 
      // Sets the port number
      ods.setPortNumber(new Integer((String)prop.get("Port")).intValue());
 
      // Sets the user name
      ods.setUser((String)prop.get("UserName"));
 
      // Sets the password
      ods.setPassword((String)prop.get("Password"));
      connection=ods.getConnection();
      gui.putStatus(" Connected to " + prop.get("SID") +
                    " Database as " + prop.get("UserName"));
 
    } catch(SQLException ex) { // Trap SQL errors
        System.out.println(
                     "Error in Connecting to the Database "+'\n'+ex.toString());
    } catch(IOException ex) { // Trap SQL errors
        System.out.println(
                     "Error in reading the properties file "+'\n'+ex.toString());
    }
 
    // Return false if failed to obtain connection object
    if( connection != null )
        return true;
        
    return false;
  }
  
  /**
   * prefetchRows() method is implemented to select records from the Database
   * in DEFAULT PRE-FETCH, WITH ROW PRE-FETCH =50 and WITH ROW PRE-FETCH =100.
   **/
  
  public void prefetchRows(int prefetchValue) {
    Integer timetaken = null; // Holds the execution time
    numFetchedRows = 0;
    
    // The SQL query for selecting all rows from AVAILABLE_ROOM_TYPES table
    String query = "Select  * from AVAILABLE_ROOM_TYPES";
    try {
      gui.putStatus("Started selecting rows , Please wait....");
      
      // Create a PreparedStatemt based on the query in query
      PreparedStatement prepare = connection.prepareStatement(query);
      
      // Casting the PreparedStatement prepare to OraclePreperedStatement
      
      // and setting the Row Pre-fetch value as prefetchValue
      ((OraclePreparedStatement)prepare).setRowPrefetch(prefetchValue);
      
      // Obtain time at the start of retrieval
      java.util.Date bfdte = new java.util.Date();
      
      // Execute the Query in query
      ResultSet rst = prepare.executeQuery();
      
      // Populating the ResultSet Object
      while(rst.next()) {
        numFetchedRows++;
      }
      
      // Obtain time at the end of retrieval
      java.util.Date afdte = new java.util.Date();
      
      // Compute time taken
      timetaken = new Integer((int)(afdte.getTime() - bfdte.getTime()));
      gui.displayResult(prefetchValue, timetaken); // Display the results
      gui.putStatus(numFetchedRows + " Rows Selected");
      prepare.close(); // Close the preparedStatement object
    }
    catch(SQLException ex) { // Trap SQL Errors
      gui.timer.stop();
      gui.putStatus("Error in selecting rows " + '\n' + ex.toString());
    }
  }
  
  /**
   *  Count the total number of rows in the AVAILABLE_ROOM_TYPES table which is
   *  used to check the condition in TimeListener class
   **/
  
  public void countRows() {
    
    // The SQL query for counting the total number of rows
    
    // from AVAILABLE_ROOM_TYPES table
    String query = "Select count(*) from AVAILABLE_ROOM_TYPES ";
    try {
      
      // Create a SQL statement based on the query in query
      Statement stmt = connection.createStatement();
      
      // Execute the Query in query
      ResultSet result = stmt.executeQuery(query);
      
      // Populating the ResultSet object
      while(result.next()) {
        numTotalRows = result.getInt(1); // Gets the total number of rows
      }
      
      // Closing the statement object
      stmt.close();
    }
    catch(SQLException ex) { // Trap SQL Errors
      gui.putStatus("Error in selecting the rows " + '\n' + ex.toString());
    }
  }
  
  /**
   *  Closing the Database Connection Object and exit the application
   **/
  
  public void exitApplication() {
    try {
      gui.putStatus("Closing the connection....please wait.....");
      if(connection != null) {
        connection.close(); //Closing the connection object.
      }
    }
    catch(SQLException ex) { //Trap SQL Errors
      gui.putStatus(ex.toString());
    }
    System.exit(0); //Exit the aplication
  }
}
