/*
 * @author  rsundara.in
 * @version 1.0
 *
 * Development Environment        :  Oracle 9i JDeveloper
 * Name of the Application        :  ThreadSample.java
 * Creation/Modification History  :
 *
 * rsundara.in       16-Jan-1999     Created
 * Reghu             15-May-2002     Modified  
 *
 */
package oracle.otnsamples.jdbc.thread;

// Java utility classes
import java.util.Vector;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.Enumeration;

// JDBC classes
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Connection;
import java.sql.Statement;

// AWT classes
import java.awt.Dimension;
import java.awt.Toolkit;

import java.io.IOException; 

import oracle.jdbc.pool.OracleDataSource; // To get a database connection 

/**
 * This sample illustrates the use of threads in a JDBC application.
 * Java Threads is a native java feature using which, two paths of execution
 * can execute concurrently in a co-operative manner. By using threads in a
 * typical java application which performs database access and data processing,
 * one can improve performance by having two threads, one performing data
 * access, and other handling the processing of data simultaneously.
 *
 * The aim of this application is to illustrate the typical use of Java Threads
 * in a database application. It retrieves all the rows from AVAILABLE_ROOM_TYPES
 * table, displays the rows and also computes the average rate and total number
 * of rooms of each type. The database retrieval is done in one thread and
 * the display and computation in another. The computation thread also performs
 * a database operation (a join to get the hotel name from the hotel id).
 */ 
public class ThreadSample extends Thread{

  private Connection connection = null;// Holds the database connection context
  
  private int rowfetchedctr=0;     // To keep track of the no of rows fetched
  private int rowdisplayedctr=0;   // To keep track of the no of rows displayed

  // Number of rooms of each type
  private int DbleCtr=0,KingCtr=0,OrclCtr=0,OthrCtr=0,QeenCtr=0,SglbCtr=0,SuitCtr=0;

  private Vector dataholder=new Vector(); // For holding the data fetched from the db
  private Thread rowfetcherthread=null;   // Thread object for rowfetching
  private Thread dataprocessorthread=null;// Thread object for dataprocessing

  private int rate=0;          // Total rate of the rooms fetched

  // Running average rate for each type of room
  private int dblerate=0,kingrate=0,orclrate=0,othrrate=0,qeenrate=0,sglbrate=0,suitrate=0;

  private boolean suspended=false;// boolean variable to start/stop threads

  ThreadFrame gui;
  
 /**
  * Constructor. Initializes the JTable and other gui components.
  */
 public ThreadSample() {
    try {
       gui=new ThreadFrame(this);
       // Diplay the frame in the center of screen
       Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
       Dimension frameSize  = gui.getSize();
       if( frameSize.height > screenSize.height ) {
          frameSize.height = screenSize.height;
       }
       if (frameSize.width > screenSize.width) {
          frameSize.width = screenSize.width;
       }
       gui.setLocation((screenSize.width - frameSize.width) / 2, 
                      (screenSize.height - frameSize.height) / 2);
       gui.setVisible(true);
    } catch (Exception e) { // Trap Errors
      e.printStackTrace();
    }
 }

 /**
  * The main entry point to the application. Instantiates main class
  * and database connection is made here.
  */
 public static void main(String[] args)throws SQLException{
    ThreadSample TS=new ThreadSample();// Instantiates the root frame
    TS.dbConnection();                 // connects to the database
 }

 /**
  * Dispatches the gui events to the appropriate method, which performs
  * the required JDBC operations. This method is invoked when event occurs
  * in the gui (like table Selection, Button clicks etc.). This method
  * is invoked from the setupListeners section of ThreadFrame.java
  */
  public void dispatchEvent(String eventName) {

    // Dispatch Event
    if (eventName.equals("START THREADS")){
           startRowFetch();
           startDataProcess();
    }
    if(eventName.equals("SUSPEND/RESUME THREADS")){
        if(!suspended){
            gui.putStatus("Threads suspended...");
            if (rowfetcherthread != null)
                rowfetcherthread.suspend();
            dataprocessorthread.suspend();
            suspended=true;
        } else{
            gui.putStatus("Threads in Progress...");
            if(rowfetcherthread != null)
               rowfetcherthread.resume();
            dataprocessorthread.resume();
            suspended=false;
        }
    }
    if (eventName.equals("EXIT"))
      exitApplication();
 }

  /**
   * This method reads a properties file which is passed as
   * the parameter to it and load it into a java Properties 
   * object and returns it.
   */
  public static Properties loadParams( String file ) throws IOException {
    // Loads a ResourceBundle and creates Properties from it
    Properties prop = new Properties();
    ResourceBundle bundle = ResourceBundle.getBundle( file );
    Enumeration enum = bundle.getKeys();
    String key = null;
    while( enum.hasMoreElements() ) {
      key = (String)enum.nextElement();
      prop.put( key, bundle.getObject( key ) );
    }
    return prop;
  }

  /**
   * Creates a database connection object using DataSource object. Please 
   * substitute the database connection parameters with appropriate values in
   * Connection.properties file
   */
  public boolean dbConnection() {
    try {
      gui.putStatus("Trying to connect to the Database");

      // Load the properties file to get the connection information
      Properties prop = this.loadParams("Connection");

      // Create a OracleDataSource instance
      OracleDataSource ods = new OracleDataSource();

     	// Sets the driver type
      ods.setDriverType("thin");

     	// Sets the database server name
     	ods.setServerName((String)prop.get("HostName"));

     	// Sets the database name
     	ods.setDatabaseName((String)prop.get("SID"));

     	// Sets the port number
      ods.setPortNumber(new Integer((String)prop.get("Port")).intValue());

      // Sets the user name
      ods.setUser((String)prop.get("UserName"));

      // Sets the password
      ods.setPassword((String)prop.get("Password"));

      // Create a connection  object
      connection = ods.getConnection();

      gui.putStatus(" Connected to " + prop.get("SID") +
                    " Database as " + prop.get("UserName"));

    } catch(SQLException ex) { // Trap SQL errors
      gui.putStatus("Error in Connecting to the Database "+'\n'+ex.toString());
    } catch(IOException ex) { // Trap SQL errors
      gui.putStatus("Error in reading the properties file "+'\n'+ex.toString());
    }

    // Return false if failed to obtain connection object
    if( connection != null )
        return true;
        
    return false;
  }


 /**
  * Starts the thread to retrieve data from the database
  */
 public void startRowFetch(){
   if(rowfetcherthread == null){
      rowfetcherthread=new Thread(this,"RowFetcher");
      rowfetcherthread.start();
   }
 }

 /**
  * Starts the thread to process the data retrieved
  */
 public void startDataProcess(){
   if(dataprocessorthread == null){
      dataprocessorthread = new Thread(this,"DataProcessor");
      dataprocessorthread.start();
   }
 }

 /**
  * This method fetches the records from the database and stores it in a local
  * buffer
  */
 public void fetchData(){
     try{
        gui.putStatus("Threads in Progress...");
        gui.beforeFetch();

        // Retrieve the resultset for the query retrieving all rows from the
        // the AVAILABLE_ROOM_TYPES
        Statement statement=connection.createStatement();
        ResultSet rset=statement.executeQuery(
          "select avr.hot_id,avr.room_type,avr.standard_rate"+
          " from available_room_types avr");

        // Loop through the resultset and retrieve all rows
        while(rset.next()){
            rowfetchedctr++;

            // Looping through the result set & putting the data into the vector
            Vector tempholder = new Vector();
            tempholder.addElement(rset.getString(1));
            tempholder.addElement(rset.getString(2));
            tempholder.addElement(new Integer(rset.getInt(3)));
            dataholder.addElement(tempholder); //Add to data vector

            Thread.currentThread().sleep(30); // Allow display thread to catch up
        }
     }catch(Exception ex){ //Trap SQL Errors
            gui.putStatus(ex.toString());
      }
 }

 /**
  * This method obtains data from the local buffer, updates gui. Also retrieves
  * the hotel name from the database based on the hoteid present in the
  * local buffer
  */
 public void processData(){
   try{
      // The SQL statement to retrieve the hotel name
      PreparedStatement pStatement=connection.prepareStatement(
                              "select name from hotels where id = ?");

      // Loop till all rows fetched have been processed
      while(true){
        // If all rows have been processed exit
        if((rowfetcherthread==null) && (rowdisplayedctr== rowfetchedctr))
          break;

        // Wait till data is available in the local buffer
        while (rowdisplayedctr >= dataholder.size())
          Thread.currentThread().sleep(10);

          //Obtain data from the data vector for displaying them in gui
          Vector row = (Vector) dataholder.elementAt(rowdisplayedctr);
          String nam=  (String)row.elementAt(0);
          String type=(String)row.elementAt(1);
          Integer rat=(Integer)row.elementAt(2);

          // Bind Hotel ID to the prepared statement, and retrieve the resultset
          pStatement.setString(1,nam);
          ResultSet resultSet = pStatement.executeQuery();

          // Retrieve the Hotel Name from the resultset
          if (resultSet.next())
            nam = resultSet.getString(1);
          resultSet.close();

          gui.addToTableData(nam,type,rat); // Add to JTablle
          rowdisplayedctr++;
          gui.trowfetched.setText(" "+rowfetchedctr);

          int rate=rat.intValue();

         // Update running average and increment counter for room type DOUBLE
         if(type.equals("DBLE")){
           DbleCtr++;
           dblerate+=rate;
           gui.tdouble.setText(Integer.toString(DbleCtr));
           float drate=(float)dblerate/DbleCtr;
           gui.tdoublerate.setText("$ "+drate);
         }
         // Update running average and increment counter for room type KING
         if(type.equals("KING")){
          KingCtr++;
          kingrate+=rate;
          gui.tking.setText(Integer.toString(KingCtr));
          float krate=(float)kingrate/KingCtr;
          gui.tkingrate.setText("$ "+krate);
        }
        // Update running average and increment counter for room type ORCL
        if(type.equals("ORCL")){
          OrclCtr++;
          orclrate+=rate;
          gui.torcl.setText(Integer.toString(OrclCtr));
          float orate=(float)orclrate/OrclCtr;
          gui.torclrate.setText("$ "+orate);
        }
        // Update running average and increment counter for room type OTHER
        if(type.equals("OTHR")){
          OthrCtr++;
          othrrate+=rate;
          gui.tothers.setText(Integer.toString(OthrCtr));
          float otrate=(float)othrrate/OthrCtr;
          gui.tothersrate.setText("$ "+otrate);
        }
        // Update running average and increment counter for room type QUEEN
        if(type.equals("QEEN")){
          QeenCtr++;
          qeenrate+=rate;
          gui.tqueen.setText(Integer.toString(QeenCtr));
          float qrate=(float)qeenrate/QeenCtr;
          gui.tqueenrate.setText("$ "+qrate);
        }
        // Update running average and increment counter for room type SINGLE
        if(type.equals("SGLB")){
          SglbCtr++;
          sglbrate+=rate;
          gui.tsglb.setText(Integer.toString(SglbCtr));
          float tsrate=(float)sglbrate/SglbCtr;
          gui.tsglbrate.setText("$ "+tsrate);
        }
        // Update running average and increment counter for room type SUITE
        if(type.equals("SUIT")){
          SuitCtr++;
          suitrate+=rate;
          gui.tsuit.setText(Integer.toString(SuitCtr));
          float srate=(float)suitrate/SuitCtr;
          gui.tsuitrate.setText("$ "+srate);
        }
      // displaying the number of the processsed rows
      gui.trowdisplayed.setText(" "+rowdisplayedctr);

    }
    gui.putStatus("Processing Completed.");
    dataprocessorthread=null;
    pStatement.close();

   } catch(Exception ex) {   // Trap general Errors
     gui.putStatus(ex.toString());
   }
  }


 /**
  * This method calls the appropriate method for each thread
  */
  public void run(){

   // The data retrieval thread
   if(rowfetcherthread == Thread.currentThread()) {
        fetchData();
        rowfetcherthread=null;
        System.gc();
   }

   // The data processing thread
   if(dataprocessorthread==Thread.currentThread()) {
        processData();
        dataprocessorthread=null;
        System.gc();
   }
  }

  /**
   *  This method closes the connection object and exits application
   */
  public void exitApplication() {
    try{
      // Stop threads and close connection
      if (rowfetcherthread != null)
          rowfetcherthread.stop();
      if (dataprocessorthread != null)
          dataprocessorthread.stop();
      if (connection!=null)
          connection.close();

    } catch(Exception ex){ //Trap SQL Errors
          gui.putStatus(ex.toString());
    }
    System.exit(0);
 }
}

